#include "splashkit.h"

// Game window size
const int WINDOW_WIDTH = 400;
const int WINDOW_HEIGHT = 600;

// Bird properties
const int BIRD_X = 100;
const int BIRD_SIZE = 20;
const double GRAVITY = 0.5;
const double JUMP_STRENGTH = -8;

// Pipe properties
const int PIPE_WIDTH = 60;
const int PIPE_GAP = 150;
const double PIPE_SPEED = 2.0;
const int PIPE_SPACING = 250;

// Font for text
font game_font;

// Game state structure
struct GameState {
    // Bird position and movement
    double bird_y;
    double bird_velocity;
    
    // Pipe positions (x, gap_y)
    double pipe1_x, pipe1_gap_y;
    double pipe2_x, pipe2_gap_y;
    
    // Game status
    int score;
    bool game_over;
    bool started;
    bool pipe1_passed;
    bool pipe2_passed;
    
    // Constructor
    GameState() {
        bird_y = WINDOW_HEIGHT / 2;
        bird_velocity = 0;
        pipe1_x = WINDOW_WIDTH;
        pipe1_gap_y = WINDOW_HEIGHT / 2;
        pipe2_x = WINDOW_WIDTH + PIPE_SPACING;
        pipe2_gap_y = WINDOW_HEIGHT / 2;
        score = 0;
        game_over = false;
        started = false;
        pipe1_passed = false;
        pipe2_passed = false;
    }
};

// Handle input
void handle_input(GameState& game) {
    if (key_typed(SPACE_KEY)) {
        if (!game.started) {
            game.started = true;
        }
        if (!game.game_over) {
            game.bird_velocity = JUMP_STRENGTH;
        }
    }
    
    if (key_typed(SPACE_KEY) && game.game_over) {
        // Reset game
        game.bird_y = WINDOW_HEIGHT / 2;
        game.bird_velocity = 0;
        game.pipe1_x = WINDOW_WIDTH;
        game.pipe1_gap_y = WINDOW_HEIGHT / 2;
        game.pipe2_x = WINDOW_WIDTH + PIPE_SPACING;
        game.pipe2_gap_y = WINDOW_HEIGHT / 2;
        game.score = 0;
        game.game_over = false;
        game.started = false;
        game.pipe1_passed = false;
        game.pipe2_passed = false;
    }
}

// Update game logic
void update_game(GameState& game) {
    if (!game.started || game.game_over) {
        return;
    }
    
    // Apply gravity to bird
    game.bird_velocity += GRAVITY;
    game.bird_y += game.bird_velocity;
    
    // Move pipes
    game.pipe1_x -= PIPE_SPEED;
    game.pipe2_x -= PIPE_SPEED;
    
    // Reset pipes when they go off screen
    if (game.pipe1_x < -PIPE_WIDTH) {
        game.pipe1_x = WINDOW_WIDTH;
        game.pipe1_gap_y = 150 + rnd() * (WINDOW_HEIGHT - 300);
        game.pipe1_passed = false;
    }
    if (game.pipe2_x < -PIPE_WIDTH) {
        game.pipe2_x = WINDOW_WIDTH;
        game.pipe2_gap_y = 150 + rnd() * (WINDOW_HEIGHT - 300);
        game.pipe2_passed = false;
    }
    
    // Increase score when pipe crosses bird's x
    if (!game.pipe1_passed && game.pipe1_x + PIPE_WIDTH < BIRD_X) {
        game.score++;
        game.pipe1_passed = true;
    }
    if (!game.pipe2_passed && game.pipe2_x + PIPE_WIDTH < BIRD_X) {
        game.score++;
        game.pipe2_passed = true;
    }
    
    // Check collisions
    // Ground and ceiling
    if (game.bird_y < 0 || game.bird_y > WINDOW_HEIGHT - BIRD_SIZE) {
        game.game_over = true;
    }
    
    // Pipe 1 collision
    if (game.bird_y + BIRD_SIZE > game.pipe1_gap_y + PIPE_GAP/2 || 
        game.bird_y < game.pipe1_gap_y - PIPE_GAP/2) {
        if (BIRD_X + BIRD_SIZE > game.pipe1_x && 
            BIRD_X < game.pipe1_x + PIPE_WIDTH) {
            game.game_over = true;
        }
    }
    
    // Pipe 2 collision
    if (game.bird_y + BIRD_SIZE > game.pipe2_gap_y + PIPE_GAP/2 || 
        game.bird_y < game.pipe2_gap_y - PIPE_GAP/2) {
        if (BIRD_X + BIRD_SIZE > game.pipe2_x && 
            BIRD_X < game.pipe2_x + PIPE_WIDTH) {
            game.game_over = true;
        }
    }
}

// Draw everything
void draw_game(const GameState& game) {
    clear_screen(COLOR_SKY_BLUE);
    
    // Draw pipes
    // Pipe 1
    fill_rectangle(COLOR_GREEN, game.pipe1_x, 0, PIPE_WIDTH, game.pipe1_gap_y - PIPE_GAP/2);
    fill_rectangle(COLOR_GREEN, game.pipe1_x, game.pipe1_gap_y + PIPE_GAP/2, PIPE_WIDTH, WINDOW_HEIGHT);
    
    // Pipe 2
    fill_rectangle(COLOR_GREEN, game.pipe2_x, 0, PIPE_WIDTH, game.pipe2_gap_y - PIPE_GAP/2);
    fill_rectangle(COLOR_GREEN, game.pipe2_x, game.pipe2_gap_y + PIPE_GAP/2, PIPE_WIDTH, WINDOW_HEIGHT);
    
    // Draw bird
    fill_circle(COLOR_YELLOW, BIRD_X, game.bird_y, BIRD_SIZE);
    
    // Draw score
    draw_text("Score: " + std::to_string(game.score), COLOR_WHITE, game_font, 24, 10, 10);
    
    // Draw instructions
    if (!game.started) {
        draw_text("Press SPACE to start", COLOR_WHITE, game_font, 18, WINDOW_WIDTH/2 - 75, WINDOW_HEIGHT/2);
    } else if (game.game_over) {
        draw_text("GAME OVER!", COLOR_RED, game_font, 32, WINDOW_WIDTH/2 - 85, WINDOW_HEIGHT/2 - 50);
        draw_text("Press Space to restart", COLOR_WHITE, game_font, 18, WINDOW_WIDTH/2 - 85, WINDOW_HEIGHT/2);
    }
}

// Main function
int main() {
    open_window("Flappy Bird", WINDOW_WIDTH, WINDOW_HEIGHT);
    
    // Load font
    game_font = load_font("GameFont", "DMSerifText.ttf");
    
    GameState game;
    
    while (!quit_requested()) {
        process_events();
        handle_input(game);
        update_game(game);
        draw_game(game);
        refresh_screen(60);
    }
    
    close_all_windows();
    return 0;
}
